<?php
/**
 * CC WP Migration Command
 * Copyright (C) 2014-2018 CCWPMask Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

if ( class_exists( 'WP_CLI_Command' ) ) {
	class CCwpm_WP_CLI_Command extends WP_CLI_Command {
		/**
		 * Creates a new backup.
		 *
		 * ## OPTIONS
		 *
		 * [--list]
		 * Get a list of backup files
		 *
		 * Example:
		 * $ wp ccwpm backup --list
		 * +------------------------------------------------+--------------+-----------+
		 * | Backup name                                    | Date created | Size      |
		 * +------------------------------------------------+--------------+-----------+
		 * | migration-wp-20170908-152313-435.ccpress        | 4 days ago   | 536.77 MB |
		 * | migration-wp-20170908-152103-603.ccpress        | 4 days ago   | 536.77 MB |
		 * | migration-wp-20170908-152036-162.ccpress        | 4 days ago   | 536.77 MB |
		 * | migration-wp-20170908-151428-266.ccpress        | 4 days ago   | 536.77 MB |
		 * +------------------------------------------------+--------------+-----------+
		 *
		 * [--exclude-spam-comments]
		 * Do not export spam comments
		 *
		 * [--exclude-post-revisions]
		 * Do not export post revisions
		 *
		 * [--exclude-media]
		 * Do not export media library (files)
		 *
		 * [--exclude-themes]
		 * Do not export themes (files)
		 *
		 * [--exclude-inactive-themes]
		 * Do not export inactive themes (files)
		 *
		 * [--exclude-muplugins]
		 * Do not export must-use plugins (files)
		 *
		 * [--exclude-plugins]
		 * Do not export plugins (files)
		 *
		 * [--exclude-inactive-plugins]
		 * Do not export inactive plugins (files)
		 *
		 * [--exclude-cache]
		 * Do not export cache (files)
		 *
		 * [--exclude-database]
		 * Do not export database (sql)
		 *
		 * [--exclude-email-replace]
		 * Do not replace email domain (sql)
		 *
		 * [--replace "find" "replace"]
		 * Find and replace text in the database
		 *
		 * Example:
		 * $ wp ccwpm backup --replace "wp" "WordPress"
		 * Backup in progress...
		 * Backup complete.
		 * Backup location: /repos/migration/wp/wp-content/ccwpm-backups/migration-wp-20170913-095743-931.ccpress
		 *
		 * @subcommand backup
		 */
		public function backup( array $args, array $assoc_args ) {
			if ( is_multisite() ) {
				WP_CLI::error( __( 'WPI CLI is not supported in Multisite mode. Please use web interface to create a backup.', CCWPM_PLUGIN_NAME ) );
			}

			if ( ! is_dir( CCWPM_STORAGE_PATH ) ) {
				if ( ! mkdir( CCWPM_STORAGE_PATH ) ) {
					WP_CLI::error_multi_line( array(
						sprintf( __( 'CC WP Migration is not able to create <strong>%s</strong> folder.', CCWPM_PLUGIN_NAME ), CCWPM_STORAGE_PATH ),
						__( 'You will need to create this folder and grant it read/write/execute permissions (0777) for the CC WP Migration plugin to function properly.', CCWPM_PLUGIN_NAME ),
					) );
					exit;
				}
			}

			if ( ! is_dir( CCWPM_BACKUPS_PATH ) ) {
				if ( ! mkdir( CCWPM_BACKUPS_PATH ) ) {
					WP_CLI::error_multi_line( array(
						sprintf( __( 'CC WP Migration is not able to create <strong>%s</strong> folder.', CCWPM_PLUGIN_NAME ), CCWPM_BACKUPS_PATH ),
						__( 'You will need to create this folder and grant it read/write/execute permissions (0777) for the CC WP Migration plugin to function properly.', CCWPM_PLUGIN_NAME ),
					) );
					exit;
				}
			}

			$params = array();
			if ( isset( $assoc_args['list'] ) ) {
				$backups = new cli\Table;

				$backups->setHeaders( array(
					'name' => __( 'Backup name', CCWPM_PLUGIN_NAME ),
					'date' => __( 'Date created', CCWPM_PLUGIN_NAME ),
					'size' => __( 'Size', CCWPM_PLUGIN_NAME ),
				) );

				$model = new CCwpm_Backups;
				foreach ( $model->get_files() as $backup ) {
					$backups->addRow( array(
						'name' => $backup['filename'],
						'date' => sprintf( __( '%s ago', CCWPM_PLUGIN_NAME ), human_time_diff( $backup['mtime'] ) ),
						'size' => size_format( $backup['size'], 2 ),
					) );
				}

				$backups->display();
				exit;
			}

			if ( isset( $assoc_args['exclude-spam-comments'] ) ) {
				$params['options']['no_spam_comments'] = true;
			}

			if ( isset( $assoc_args['exclude-post-revisions'] ) ) {
				$params['options']['no_post_revisions'] = true;
			}

			if ( isset( $assoc_args['exclude-media'] ) ) {
				$params['options']['no_media'] = true;
			}

			if ( isset( $assoc_args['exclude-themes'] ) ) {
				$params['options']['no_themes'] = true;
			}

			if ( isset( $assoc_args['exclude-inactive-themes'] ) ) {
				$params['options']['no_inactive_themes'] = true;
			}

			if ( isset( $assoc_args['exclude-muplugins'] ) ) {
				$params['options']['no_muplugins'] = true;
			}

			if ( isset( $assoc_args['exclude-plugins'] ) ) {
				$params['options']['no_plugins'] = true;
			}

			if ( isset( $assoc_args['exclude-inactive-plugins'] ) ) {
				$params['options']['no_inactive_plugins'] = true;
			}

			if ( isset( $assoc_args['exclude-cache'] ) ) {
				$params['options']['no_cache'] = true;
			}

			if ( isset( $assoc_args['exclude-database'] ) ) {
				$params['options']['no_database'] = true;
			}

			if ( isset( $assoc_args['exclude-email-replace'] ) ) {
				$params['options']['no_email_replace'] = true;
			}

			if ( isset( $assoc_args['replace'] ) ) {
				for ( $i = 0; $i < count( $args ); $i++ ) {
					if ( isset( $args[ $i ] ) && isset( $args[ $i + 1 ] ) ) {
						$params['options']['replace']['old_value'][] = $args[ $i ];
						$params['options']['replace']['new_value'][] = $args[ $i + 1 ];
					}
				}
			}

			WP_CLI::log( 'Backup in progress...' );

			try {

				// Disable completed timeout
				add_filter( 'ccwpm_completed_timeout', '__return_zero' );

				// Remove filters
				remove_filter( 'ccwpm_export', 'CCwpm_Export_Clean::execute', 300 );

				// Run filters
				$params = apply_filters( 'ccwpm_export', $params );

			} catch ( Exception $e ) {
				WP_CLI::error( $e->getMessage() );
			}

			// Clean storage folder
			CCwpm_Directory::delete( ccwpm_storage_path( $params ) );

			WP_CLI::log( __( 'Backup complete.', CCWPM_PLUGIN_NAME ) );
			WP_CLI::log( sprintf( __( 'Backup location: %s', CCWPM_PLUGIN_NAME ), ccwpm_backup_path( $params ) ) );
		}

		/**
		 * Restores a backup.
		 *
		 * Example:
		 * $ wp ccwpm restore migration-wp-20170913-095743-931.ccpress
		 * Restore in progress...
		 * Restore complete.
		 *
		 * @subcommand restore
		 */
		public function restore( array $args, array $assoc_args ) {
			if ( is_multisite() ) {
				WP_CLI::error( __( 'WPI CLI is not supported in Multisite mode. Please use web interface to restore a backup.', CCWPM_PLUGIN_NAME ) );
			}

			if ( ! is_dir( CCWPM_STORAGE_PATH ) ) {
				if ( ! mkdir( CCWPM_STORAGE_PATH ) ) {
					WP_CLI::error_multi_line( array(
						sprintf( __( 'CC WP Migration is not able to create <strong>%s</strong> folder.', CCWPM_PLUGIN_NAME ), CCWPM_STORAGE_PATH ),
						__( 'You will need to create this folder and grant it read/write/execute permissions (0777) for the CC WP Migration plugin to function properly.', CCWPM_PLUGIN_NAME ),
					) );
					exit;
				}
			}

			$params = array();
			if ( isset( $args[0] ) && is_file( ccwpm_backup_path( array( 'archive' => $args[0] ) ) ) ) {
				$params = array(
					'storage'              => ccwpm_storage_folder(),
					'archive'              => $args[0],
					'ccwpm_manual_restore' => true,
				);
			} else {
				WP_CLI::error_multi_line( array(
					__( 'A valid backup name must be provided in order to proceed with the restore process.', CCWPM_PLUGIN_NAME ),
					__( 'Example: wp ccwpm restore migration-wp-20170913-095743-931.ccpress', CCWPM_PLUGIN_NAME ),
				) );
				exit;
			}

			WP_CLI::log( 'Restore in progress...' );

			try {

				// Disable completed timeout
				add_filter( 'ccwpm_completed_timeout', '__return_zero' );

				// Remove filters
				remove_filter( 'ccwpm_import', 'CCwpm_Import_Upload::execute', 5 );
				remove_filter( 'ccwpm_import', 'CCwpm_Import_Confirm::execute', 100 );
				remove_filter( 'ccwpm_import', 'CCwpm_Import_Clean::execute', 400 );

				// Run filters
				$params = apply_filters( 'ccwpm_import', $params );

			} catch ( Exception $e ) {
				WP_CLI::error( $e->getMessage() );
			}

			// Clean storage folder
			CCwpm_Directory::delete( ccwpm_storage_path( $params ) );

			WP_CLI::log( 'Restore complete.' );
		}
			/**
		 * Get a list of backup files.
		 *
		 * ## EXAMPLES
		 *
		 * $ wp ccwpm list-backups
		 * +------------------------------------------------+--------------+-----------+
		 * | Backup name                                    | Date created | Size      |
		 * +------------------------------------------------+--------------+-----------+
		 * | migration-wp-20170908-152313-435.ccpress        | 4 days ago   | 536.77 MB |
		 * | migration-wp-20170908-152103-603.ccpress        | 4 days ago   | 536.77 MB |
		 * | migration-wp-20170908-152036-162.ccpress        | 4 days ago   | 536.77 MB |
		 * | migration-wp-20170908-151428-266.ccpress        | 4 days ago   | 536.77 MB |
		 * +------------------------------------------------+--------------+-----------+
		 *
		 * @subcommand list-backups
		 */
		public function list_backups( array $args, array $assoc_args ) {
			$backups = new cli\Table;

			$backups->setHeaders(
				array(
					'name' => __( 'Backup name', CCWPM_PLUGIN_NAME ),
					'date' => __( 'Date created', CCWPM_PLUGIN_NAME ),
					'size' => __( 'Size', CCWPM_PLUGIN_NAME ),
				)
			);

			$model = new CCwpm_Backups;
			foreach ( $model->get_files() as $backup ) {
				$backups->addRow(
					array(
						'name' => $backup['filename'],
						'date' => sprintf( __( '%s ago', CCWPM_PLUGIN_NAME ), human_time_diff( $backup['mtime'] ) ),
						'size' => ccwpm_size_format( $backup['size'], 2 ),
					)
				);
			}

			$backups->display();
		}
	
	}
}
if ( class_exists( 'WP_CLI' ) ) {
	WP_CLI::add_command( 'ccwpm', 'CCwpm_WP_CLI_Command', array( 'short_desc' => 'CC WP Migration Command' ) );
}
